%% Caratteristica_IV.m — Fit per curva (n, Rs, Rsh) indipendenti per ogni G - stimare i parametri del diodo per ciascun irraggiamento
% Obiettivo: data una famiglia di curve I–V sperimentali (una per ogni irraggiamento G)
% stimare per ciascuna curva, i parametri del diodo equivalente [n, Rs, Rsh]  
% Parametri da stimare per ogni curva minimizzando lo scarto tra corrente simulata e corrente misurata.            
% Criterio di ottimizzazione (least-squares)
% Nota: i blocchi Simulink leggono le variabili dal base workspace (assegnate via assignin). 
% Il modello attinge dal base workspace.
% Il modello atteso è 'PV_SC_Model_IV' con segnali VoutPV e IoutPV disponibili in out. 


clear;                               % Pulisce le variabili nel workspace corrente
%clc;                                % (Opzionale) Pulisce la Command Window

% =================== Impostazioni base ===================
Cartella_con_dati = fullfile(pwd, 'Dati Sperimentali');  % cartella dove leggere i .txt (usa path relativo alla cartella corrente)
modelName = 'PV_SC_Model_IV';                            % nome del modello Simulink da simulare

% Peso (debole) per avvicinare anche il Fill Factor [adim]
lambdaFF = 0.08;      % tipicamente 0.01–0.2; 0 disattiva la penalità sul FF   % Regolarizzazione su FF nel residuo

% Bounds e guess "ampi" (usati per curve in luce)
% x = [n, Rs[Ω], Rsh[Ω]]
lb         = [1.0,  0,     1e3];     % Limiti inferiori per n, Rs, Rsh nelle curve illuminate
ub         = [3.0,  50,    1e10];    % Limiti superiori per n, Rs, Rsh nelle curve illuminate
x0_default = [1.3,  0.5,   1e6];     % Guess di partenza di default per le curve illuminate

% Bounds più "conduttivi" SOLO per G=0 (dark IV), per evitare Rsh→∞ e Rs→0
lb_dark = [1.2,  5e-3,  1e4];        % Limiti inferiori per il caso al buio
ub_dark = [3.0,    1,   1e7];        % Limiti superiori per il caso al buio
x0      = [1.8,  0.2,   1e6];        % Guess iniziale (non usata direttamente più sotto, tenuta per riferimento)

% Guess specifici (una riga per dataset; se i dataset sono di più, si riusa l’ultima riga)
x0_list = [                           % Matrice di guess per curva (k-esima riga -> k-esimo dataset)
    1.10   0.20   1e5                 % Guess per dataset 1
    1.50   1.00   1e7                 % Guess per dataset 2
    2.00   5.00   1e8                 % Guess per dataset 3
    1.80   0.10   5e6                 % Guess per dataset 4 (riusata per i successivi se >4 file)
];

% Opzioni ottimizzatore lsqnonlin:
% - Display 'iter-detailed' => utile in debug; usa 'final' per output più pulito
% - Tolleranze strette ma non estreme per evitare iterazioni inutili
% - Forward diff stabile; pchip in interp1 later evita oscillazioni
opts = optimoptions('lsqnonlin', ...           % Crea struttura opzioni per lsqnonlin
    'Display','final', ...                     % Mostra solo il riepilogo finale
    'MaxFunctionEvaluations', 1e4, ...         % Numero massimo di valutazioni della funzione
    'MaxIterations',            200, ...       % Numero massimo di iterazioni
    'FunctionTolerance',       1e-9, ...       % Tolleranza sul valore della funzione obiettivo
    'StepTolerance',           1e-8, ...       % Tolleranza sul passo dell’algoritmo
    'FiniteDifferenceType','forward');         % Differenze finite in avanti per il gradiente numerico

% =================== Import dati & parametri iniziali ===================
% Lettura file .txt, costruzione struttura dati e tabella riassuntiva
[Dati_sperimentali_da_file, Tabella_riassuntiva] = importaDatiSperimentali(Cartella_con_dati);  % Legge e parsifica tutti i .txt

% --- PREPARA LUT per il modello (lette da una 1-D Lookup Table nel modello)
% La LUT mappa G -> (Isc, Voc) per consolidare coerenza tra fit e Simulink
G_all   = [Dati_sperimentali_da_file.G_value]';   % Vettore di tutti i valori G dai file
Isc_all = [Dati_sperimentali_da_file.Isc_A]';     % Vettore Isc associati ai dataset
Voc_all = [Dati_sperimentali_da_file.Voc_V]';     % Vettore Voc associati ai dataset

% Rimuovi duplicati su G e garantisci ordine (qui 'stable' preserva l’ordine di apparizione)
[ G_bp, idx ] = unique(G_all, 'stable');          % Breakpoints G univoci per la LUT e indici corrispondenti
Isc_tab = Isc_all(idx);                           % Tabella Isc allineata ai breakpoints univoci
Voc_tab = Voc_all(idx);                           % Tabella Voc allineata ai breakpoints univoci

% Esponi LUT nel base workspace (il modello le legge da qui)
assignin('base','G_bp',    G_bp);                 % Mette G_bp nel base workspace
assignin('base','Isc_tab', Isc_tab);              % Mette Isc_tab nel base workspace
assignin('base','Voc_tab', Voc_tab);              % Mette Voc_tab nel base workspace

disp('✓ Dati sperimentali letti e caricati correttamente.');  % Messaggio di stato

% Inizializza parametri di default in base (usati da blocchi/maschere se necessario)
setInitialParameters();                           % Popola variabili base (Tref, q, ecc.) per il modello

% Numero di dataset (curve) disponibili
nSets = numel(Dati_sperimentali_da_file);         % Conta quante curve sono state trovate
if nSets == 0                                      % Se nessun dataset è presente
    error('Nessun dataset trovato in "%s".', Cartella_con_dati);  % Interrompe con errore
end

% =================== Fit per curva ===================
% Prealloc struttura risultati per efficienza e ordine
fitTemplate = struct( ...                          % Template con tutti i campi risultati
    'G_label',"", 'G_value',NaN, ...
    'n',NaN, 'Rs',NaN, 'Rsh',NaN, ...
    'Voc',NaN, 'Isc',NaN, ...
    'FF_fit',NaN, 'FF_exp',NaN, ...
    'exitflag',NaN, 'rmse',NaN, ...
    'Vexp',[], 'Iexp',[], 'Vsim',[], 'Isim',[]);
Fitted_Parameters = repmat(fitTemplate, nSets, 1); % Prealloca array di struct dei risultati

% Costante grafica e filtro stabilità (riusate più volte)
VMIN_PLOT = 20e-3;   % Evita V~0 in grafico (per chiarezza visiva)
VMIN_FIT  = 20e-3;   % Evita V~0 nel fit (stabilizza interpolazioni/pesi)

for k = 1:nSets                                     % Cicla su ciascun dataset/curva
    ds = Dati_sperimentali_da_file(k);              % Estrae la k-esima curva

    % --- Dati sperimentali (usa solo V>=0 e valori finiti)
    Vexp = ds.Voltage_V(:);                         % Vettore tensione sperimentale
    Iexp = ds.Current_A(:);                         % Vettore corrente sperimentale
    mexp = isfinite(Vexp) & isfinite(Iexp) & Vexp >= 0;  % Maschera validi: numeri finiti e V>=0
    Vexp = Vexp(mexp);                              % Applica maschera a V
    Iexp = Iexp(mexp);                              % Applica maschera a I

    if numel(Vexp) < 5                              % Controllo minimo numero di punti utili
        warning('Dataset %d: punti utili insufficienti (<5). Salto curva.', k);  % Avvisa e salta
        continue;                                   % Passa al dataset successivo
    end

    % --- Seleziona guess per questa curva: riga k di x0_list, altrimenti ultima riga
    x0 = x0_list(min(k, size(x0_list,1)), :);       % Prende la riga corrispondente o l’ultima se k eccede
    if any(~isfinite(x0)), x0 = x0_default; end     % Se la riga contiene non-finiti, fallback al default

    % --- Distinzione G=0 (dark) vs luce: pesi e bounds diversi
    isDark = (ds.G_value <= 0);                     % Flag: curva al buio se G<=0
    if isDark
        lambdaFF_local = 0;                         % Nessuna penalità FF in dark
        Isc_eff        = max(1e-12, 1e-6*max(Isc_tab));  % Piccolo Isc effettivo per scalare FF/parametri
        %Isc_eff        = max(abs(ds.Isc_A), 1e-9); % Alternativa: usa Isc dal file con epsilon
        x0             = [1.8, 1.0, 1e6];          % Guess più robusto per il dark
        lb_use         = lb_dark;                   % Limiti inferiori per dark
        ub_use         = ub_dark;                   % Limiti superiori per dark
    else
        lambdaFF_local = lambdaFF;                  % In luce, usa penalità FF
        Isc_eff        = ds.Isc_A;                  % Isc reale per la curva in luce
        lb_use         = lb;                        % Limiti inferiori per luce
        ub_use         = ub;                        % Limiti superiori per luce
    end
   
    % --- Residuo di fit: scarto sui punti + eventuale penalità FF
    resfun = @(x) residual_IV_for_fit_percurve( ... % Handle funzione residuo per lsqnonlin
                    x, modelName, Vexp, Iexp, ...
                    ds.G_value, ds.Voc_V, Isc_eff, lambdaFF_local, VMIN_FIT);

    % --- Ottimizzazione con lsqnonlin
    % Se fallisce con x0, riprova con x0_default (robustezza)
    try
        evalc('[xh, resnorm, r, exitflag, output] = lsqnonlin(resfun, x0, lb_use, ub_use, opts);'); % Esegue lsqnonlin sopprimendo output a video
    catch
        evalc('[xh, resnorm, r, exitflag, output] = lsqnonlin(resfun, x0_default, lb_use, ub_use, opts);'); % Retry con guess default
    end

    iters = NaN;                                    % Inizializza contatore iterazioni a NaN
    if exist('output','var') && isfield(output,'iterations') % Se 'output' ha il campo iterations
        iters = output.iterations;                  % Leggi numero di iterazioni eseguite
    end

    % --- Metriche
    n_fit  = xh(1);  Rs_fit = xh(2);  Rsh_fit = xh(3);   % Parametri stimati dal fit
    RMSE   = sqrt(mean(r.^2, 'omitnan'));                % Radice errore quadratico medio sui residui

    % --- Simulazione finale con parametri stimati (stesso Isc_eff usato nel fit)
    [Vsim, Isim] = runSimGetVI(modelName, ds.G_value, ds.Voc_V, Isc_eff, n_fit, Rs_fit, Rsh_fit); % Simula con parametri stimati
    [Vsim, Isim] = uniqueXYavg(Vsim, Isim);                 % Ordina e media duplicati per stabilità

    % --- Fill Factor: da curva simulata vs da valori sperimentali (se disponibili)
    if isDark
        %FF_fit = NaN;                                    % Opzione: non calcolare FF in dark (senza significato fisico)
        FF_fit = ff_from_curve(Vsim, Isim, ds.Voc_V, max(abs(ds.Isc_A), 1e-12)); % Calcolo FF anche in dark (solo indicativo)
    else
        FF_fit = ff_from_curve(Vsim, Isim, ds.Voc_V, ds.Isc_A);   % FF dalla curva simulata in luce
    end

    FF_exp = NaN;                                        % Inizializza FF sperimentale
    if isfinite(ds.Vmax_V) && isfinite(ds.Imax_A) && isfinite(ds.Voc_V) && isfinite(ds.Isc_A) ...
            && ds.Voc_V > 0 && abs(ds.Isc_A) > 1e-12     % Se sono disponibili Voc/Isc e punto di massima potenza
        FF_exp = abs(ds.Vmax_V * ds.Imax_A) / (ds.Voc_V * abs(ds.Isc_A)); % FF sperimentale da Vmax*Imax
    elseif isfinite(ds.Pmax_W) && isfinite(ds.Voc_V) && isfinite(ds.Isc_A) ...
            && ds.Voc_V > 0 && abs(ds.Isc_A) > 1e-12     % In alternativa, se c’è Pmax
        FF_exp = abs(ds.Pmax_W) / (ds.Voc_V * abs(ds.Isc_A));               % FF sperimentale da Pmax
    end

    % --- Salvataggio risultati
    Fitted_Parameters(k).G_label  = string(ds.G_label);  % Etichetta G testuale
    Fitted_Parameters(k).G_value  = ds.G_value;          % Valore G numerico
    Fitted_Parameters(k).n        = n_fit;               % Parametro n stimato
    Fitted_Parameters(k).Rs       = Rs_fit;              % Parametro Rs stimato
    Fitted_Parameters(k).Rsh      = Rsh_fit;             % Parametro Rsh stimato
    Fitted_Parameters(k).Voc      = ds.Voc_V;            % Voc dal dataset
    Fitted_Parameters(k).Isc      = ds.Isc_A;            % Isc dal dataset
    Fitted_Parameters(k).exitflag = exitflag;            % Codice di uscita dell’ottimizzatore
    Fitted_Parameters(k).rmse     = RMSE;                % RMSE sui residui
    Fitted_Parameters(k).iterations = iters;             % Iterazioni eseguite
    Fitted_Parameters(k).Vexp     = Vexp;                % Vettore V sperimentale (filtrato)
    Fitted_Parameters(k).Iexp     = Iexp;                % Vettore I sperimentale (filtrato)
    Fitted_Parameters(k).Vsim     = Vsim;                % Vettore V simulato
    Fitted_Parameters(k).Isim     = Isim;                % Vettore I simulato
    Fitted_Parameters(k).FF_fit   = FF_fit;              % FF da curva simulata
    Fitted_Parameters(k).FF_exp   = FF_exp;              % FF da dati sperimentali

    % --- Riepilogo sintetico a video
    motivo_stop = '';                                    % Inizializza stringa motivo stop
    if exist('output','var') && isfield(output,'message') && ~isempty(output.message) % Se è presente un messaggio di stop
        msg_lines = regexp(strtrim(output.message), '\r?\n', 'split'); % Spezza su righe
        motivo_stop = msg_lines{1};                      % Prendi la prima riga (riassunto)
    end
    
    fprintf('✓ G = %-6g W/m^2 | iterazioni = %-3d | Resnorm = %.3e | RMSE = %.3e | Stop: %s\n', ...
            ds.G_value, iters, resnorm, RMSE, motivo_stop);  % Stampa riga di riepilogo per la curva k

end

disp('✓ I parametri del fitting sono stati salvati in "Fitted_Parameters".');  % Conferma salvataggio risultati
fprintf(  '\n');                                                               % Riga vuota per leggibilità

% --- Metti nel workspace + tabella per analisi esterna
Fitted_Parameters_Table = struct2table(Fitted_Parameters);                     % Converte struct in tabella
assignin('base','Fitted_Parameters',       Fitted_Parameters);                 % Esporta struct nel base workspace
assignin('base','Fitted_Parameters_Table', Fitted_Parameters_Table);           % Esporta tabella nel base workspace

% =================== Grafici: Confronto I–V (Sperimentale vs Simulata) ===================
figure('Name','Confronto I–V (Sperimentale vs Simulata - fit)','Color','w');  % Crea figura dedicata
tiledlayout('flow','Padding','compact','TileSpacing','compact');               % Layout a tiles con spazi compatti
for k = 1:nSets                                                                % Cicla su tutte le curve fitted
    fp = Fitted_Parameters(k);                                                 % Accorcia riferimento
    if isempty(fp.Vsim), continue; end                                         % Salta se la curva simulata è vuota

    nexttile; hold on; grid on; box on;                                        % Nuovo riquadro, abilita griglia e riquadro

    mexpP = isfinite(fp.Vexp) & isfinite(fp.Iexp) & fp.Vexp >= VMIN_PLOT;      % Maschera dati sperimentali da plottare
    msimP = isfinite(fp.Vsim) & isfinite(fp.Isim) & fp.Vsim >= VMIN_PLOT;      % Maschera dati simulati da plottare

    plot(fp.Vexp(mexpP), fp.Iexp(mexpP), 'o', 'MarkerSize',4, 'DisplayName','Sperimentale'); % Plot punti sperimentali
    plot(fp.Vsim(msimP), fp.Isim(msimP), '-', 'LineWidth',1.4, 'DisplayName','Simulata (fit)'); % Plot curva simulata

    xlabel('V [V]'); ylabel('I [A]');                                          % Etichette assi
    title(sprintf('G = %s', fp.G_label));                                      % Titolo tile con etichetta G
    legend('Location','best');                                                 % Legenda posizionata automaticamente
end

% =================== Stampa tabella riassuntiva in Command Window ===================
fprintf('\nCaso  G [W/m^2]   n       Rs [Ω]      Rsh [Ω]     Voc [V]    Isc [A]    FF (fit)    Iter\n'); % Intestazione tabella
fprintf(  '-------------------------------------------------------------------------------------------\n'); % Riga separatrice
for k = 1:nSets                                                                % Cicla sui risultati
    fp = Fitted_Parameters(k);                                                 % Alias
    if ~isfinite(fp.n), continue; end                                          % Salta se fit non valido
    
    % FF: vuota se G<=0 (dark)
    if isfinite(fp.G_value) && fp.G_value <= 0                                 % Se dark
        ffStr = '';                                                            % Non stampare FF
    else
        ffStr = '—';                                                           % Placeholder
        if isfinite(fp.FF_fit), ffStr = sprintf('%.4g', fp.FF_fit); end        % Sostituisci con FF numerico se presente
    end
    
    iterStr = '—';                                                             % Placeholder iterazioni
    if isfield(fp,'iterations') && isfinite(fp.iterations)                     % Se disponibile
        iterStr = sprintf('%d', fp.iterations);                                % Stampa numero iterazioni
    end
    fprintf('%-4d  %-10.1f %-7.3f %-11.4e %-11.4e %-9.4f %-9.3e %-10s %5s', ... % Riga formattata con valori principali
        k, fp.G_value, fp.n, fp.Rs, fp.Rsh, fp.Voc, fp.Isc, ffStr, iterStr);
    if isfinite(fp.FF_exp) && ~(isfinite(fp.G_value) && fp.G_value<=0)         % Se FF exp valido e non dark
        fprintf('  (FF exp: %.4f)', fp.FF_exp);                                 % Aggiungi FF sperimentale
    end
    fprintf('\n');                                                             % A capo
end
fprintf('-------------------------------------------------------------------------------------------\n'); % Riga chiusura tabella

open_system(fullfile(pwd,'PV_SC_Model_IV.slx'));                               % Apre il modello Simulink a fine script

%% =================== FUNZIONI LOCALI ===================
function setInitialParameters()
% setInitialParameters — Popola nel base workspace i parametri default richiesti dal modello. % Descrizione funzione helper
% Motivazione: molte maschere/From Workspace leggono direttamente dal base; avere default
%              sensati evita errori quando il modello viene lanciato standalone.

    P.Tref  = 298.15;      P.Kbolz = 1.380649e-23;  P.q = 1.602176634e-19;   % Costanti fisiche base
    P.Ki    = 1.3e-3;      P.Eg    = 1.12;          P.Rs  = 1e-4;            % Parametri PV generici
    P.Rsh   = 1e3;         P.NsPV  = 1;             P.NpPV = 1;               % Configurazione stringa PV
    P.n     = 1.2;         P.G     = 1;                                         % Fattore idealità e G default
    P.A_cm2 = 1.44;        P.A     = P.A_cm2*1e-4;                              % Area in cm^2 e in m^2
    P.Isc   = 5.711;       P.Voc   = 0.577;         P.Top  = P.Tref;           % Isc, Voc, temperatura operativa
    P.Irs0 = 1e-12;                                                               % Corrente di saturazione di riferimento

    % Parametri supercap (se il modello li richiede; altrimenti restano innocui)
    P.R1=16.9; P.C0=0.04; P.Cv=0.0061; P.R2=1000; P.C2=0.31; P.NpSC=1; P.NsSC=1; % Parametri SC

    % Pubblica tutto nel base workspace
    fn = fieldnames(P);                                                         % Lista dei nomi campo
    for i = 1:numel(fn)                                                         % Cicla su ogni campo
        assignin('base', fn{i}, P.(fn{i}));                                     % Assegna nel base workspace
    end
end

function [data, summary] = importaDatiSperimentali(folderPath)
% importaDatiSperimentali — Legge tutti i .txt nella cartella e costruisce una struct per curva. % Descrizione
% Formato atteso:
%   - header con righe tipo "Area (cm^2): <val>", "Isc (A): <val>", "Voc (V): <val>", ecc.  % Spiega l’header atteso
%   - tabella con colonne "Voltage (V)" e "Current density (mA/cm^2)" (titoli presenti su una riga) % Colonne dati attese
% Ritorna:
%   data   — struct array con campi utili alla simulazione/fit                            % Output 1
%   summary— tabella riassuntiva senza le colonne lunghe (serie V/J/I)                   % Output 2

    if nargin<1 || isempty(folderPath), folderPath = fullfile(pwd,'Dati Sperimentali'); end % Default path se non fornito
    if ~isfolder(folderPath), error('Cartella "%s" non trovata.', folderPath); end          % Validazione cartella

    files = dir(fullfile(folderPath,'*.txt'));                                              % Elenco dei .txt nella cartella
    if isempty(files), data=struct([]); summary=table(); return; end                        % Se non ci sono file, ritorna vuoto

    template = struct('FileName',"",'G_label',"",'G_value',NaN,'Area_cm2',NaN,'Isc_A',NaN, ... % Template per ogni file
        'Voc_V',NaN,'Pmax_W',NaN,'Imax_A',NaN,'Vmax_V',NaN,'FF',NaN,'Efficiency_pct',NaN, ...
        'Jsc_mAcm2',NaN,'Voltage_V',[],'CurrentDensity_mAcm2',[],'Current_A',[]);

    nFiles = numel(files);             % Numero file trovati
    data   = repmat(template, nFiles, 1);  % Preallocazione array struct
    w = 0;                              % Contatore reali caricati (per eventuali skip)

    headerMap = {                       % Mappa “testo header file” -> “nome campo nella struct”
        'Area (cm^2)','Area_cm2'
        'Isc (A)','Isc_A'
        'Voc (V)','Voc_V'
        'Pmax (W)','Pmax_W'
        'Imax (A)','Imax_A'
        'Vmax (V)','Vmax_V'
        'FF','FF'
        'Efficiency (%)','Efficiency_pct'
        'Jsc (mA/cm^2)','Jsc_mAcm2'};

    for k = 1:numel(files)              % Cicla su ogni file .txt
        fname  = files(k).name;         % Nome del file
        fpath  = fullfile(files(k).folder, fname); % Percorso completo del file
        Glabel = extractGlabelFromFileName(fname); % Estrae etichetta G dal nome file
        Gvalue = labelToDouble(Glabel); % Converte etichetta G in valore numerico

        lines = strip(readlines(fpath));   % legge tutto il file come vettore di righe (stringhe) rimuovendo spazi ai bordi

        % --- Parse header chiave:valore
        H = struct();                   % Struct temporanea per header estratti
        for i = 1:size(headerMap,1)     % Cicla su ogni chiave attesa
            H.(headerMap{i,2}) = findHeaderValue(lines, headerMap{i,1}); % Estrae il valore numerico associato
        end

        % --- Trova intestazione delle colonne dati (Voltage, Current density)
        hdrIdx = find(contains(lines,"Voltage (V)",'IgnoreCase',true) & ...    % Cerca riga con entrambe le parole chiave
                      contains(lines,"Current density",'IgnoreCase',true), 1, 'first');
        if isempty(hdrIdx)                % Se intestazione non trovata
            error('Header colonne non trovato in "%s". Attesi "Voltage (V)" e "Current density".', fname); % Errore bloccante
        end

        % --- Estrai blocco dati e converti in numeric
        blk = lines(hdrIdx+1:end);        % Prende tutte le righe dopo l’intestazione
        blk = blk(~(blk==""));            % Elimina righe vuote

        V = []; J = [];                   % Prepara vettori per Voltage e Current density
        for r = 1:numel(blk)              % Cicla su ogni riga dati
            row    = strrep(char(blk(r)), ',', '.');       % Supporta decimale con virgola -> punto
            tokens = regexp(row,'[-+0-9eE\.]+','match');   % Estrae tutte le “parole” numeriche
            if numel(tokens) >= 2                           % Se ci sono almeno due numeri
                v = str2double(tokens{1});                 % Primo numero = tensione
                j = str2double(tokens{2});                 % Secondo numero = densità di corrente
                if isfinite(v) && isfinite(j)              % Valori validi?
                    V(end+1,1) = v; %#ok<AGROW>            % Accoda a V (colonna)
                    J(end+1,1) = j; %#ok<AGROW>            % Accoda a J (colonna)
                end
            end
        end

        % --- Costruisci entry
        entry = template;                 % Parte dal template vuoto
        entry.FileName = fname;           % Salva nome file
        entry.G_label  = Glabel;          % Etichetta G
        entry.G_value  = Gvalue;          % Valore G numerico
        entry.Area_cm2 = H.Area_cm2;      % Area dal header
        entry.Isc_A    = H.Isc_A;         % Isc dal header
        entry.Voc_V    = H.Voc_V;         % Voc dal header
        entry.Pmax_W   = H.Pmax_W;        % Pmax dal header
        entry.Imax_A   = H.Imax_A;        % Imax dal header
        entry.Vmax_V   = H.Vmax_V;        % Vmax dal header
        entry.FF       = H.FF;            % FF dal header
        entry.Efficiency_pct = H.Efficiency_pct; % Efficienza dal header
        entry.Jsc_mAcm2      = H.Jsc_mAcm2;      % Jsc dal header
        entry.Voltage_V           = V;           % Colonna tensioni
        entry.CurrentDensity_mAcm2= J;           % Colonna densità corrente

        % Corrente assoluta I [A] = J [mA/cm^2] * Area[cm^2] * 1e-3
        if isfinite(entry.Area_cm2)                   % Se area valida
            entry.Current_A = entry.CurrentDensity_mAcm2 * (entry.Area_cm2 * 1e-3); % Converte J in I assoluta
        else
            entry.Current_A = nan(size(entry.CurrentDensity_mAcm2));                % Altrimenti NaN
        end

        w = w + 1;                        % Incrementa contatore effettivi
        data(w,1) = entry;                % Inserisce entry nell’array struct
    end

    if w < nFiles                          % Se alcuni file sono stati scartati
        data = data(1:w,1);                % Ridimensiona l’array ai soli caricati
    end

    % --- Ordina per G crescente e costruisci summary
    summary = struct2table(rmfield(data,{'Voltage_V','CurrentDensity_mAcm2','Current_A'})); % Tabella senza serie lunghe
    [~,ord] = sort(summary.G_value);       % Ordina per valore G
    summary = summary(ord,:);              % Riordina tabella
    data    = data(ord);                   % Riordina struct coerentemente
end

function G_label = extractGlabelFromFileName(fname)
% Estrae stringa tipo "X,YYY W" dal nome file se presente; altrimenti "N/D". % Descrizione
    tok = regexp(string(fname), "G\s*([0-9]+(?:[.,][0-9]+)?)\s*W", 'tokens', 'ignorecase'); % Cerca pattern "G <numero> W"
    if ~isempty(tok) && ~isempty(tok{1})    % Se trovato
        numTxt = string(tok{1}{1});         % Prende la parte numerica come stringa
        if contains(numTxt,'.'), numTxt = replace(numTxt,'.',','); end % Normalizza con virgola nell’etichetta
        G_label = numTxt + " W";            % Ritorna etichetta tipo "X,YYY W"
    else
        G_label = "N/D";                    % Non disponibile
    end
end

function val = labelToDouble(G_label)
% Converte "X,YYY W" -> double X.YYY (W/m^2 equivalenti in questa etichetta) % Descrizione
    s = upper(strrep(G_label,' ', ''));     % Rimuove spazi e porta a maiuscolo
    s = erase(s,"W");                       % Elimina la lettera W
    s = strrep(s, ',', '.');                % Sostituisce virgola con punto per conversione numerica
    val = str2double(s);                    % Converte in double (NaN se fallisce)
end

function v = findHeaderValue(lines, key)
% Cerca la riga che contiene 'key' e ne estrae l’ultimo numero (supporta , e .) % Descrizione
    idx = find(contains(lines, key, 'IgnoreCase', true), 1, 'first'); % Trova la prima riga che contiene la chiave
    if isempty(idx), v = NaN; return; end     % Se non trovata, ritorna NaN
    row    = strrep(char(lines(idx)), ',', '.');      % Normalizza decimale
    tokens = regexp(row,'[-+0-9eE\.]+','match');      % Estrae tutti i numeri nella riga
    if isempty(tokens), v = NaN; else, v = str2double(tokens{end}); if ~isfinite(v), v = NaN; end, end % Prende l’ultimo numero
end

function [Vsim, Isim] = runSimGetVI(modelName, Gval, Voc, Isc, n_id, Rs, Rsh)
% runSimGetVI — Esegue il modello Simulink con parametri fissati e ritorna i vettori V,I. % Descrizione high-level
% Note importanti:
%   - Il modello deve esporre i segnali 'VoutPV' e 'IoutPV' negli Outputs (out).          % Requisito segnali
%   - Le variabili G, Voc, Isc, n, Rs, Rsh sono lette nel modello (From Workspace/Mask).  % Parametri passati via base
%   - La variabile 'G' viene passata come [t u] per compatibilità con From Workspace.      % Formato segnale G

    % --- Controllo di presenza LUT nel base workspace (usate da Lookup nel modello)
    needLUT = @(v) evalin('base', sprintf('exist(''%s'',''var'')', v)) ~= 1;   % Funzione anonima: verifica esistenza variabile
    if needLUT('G_bp') || needLUT('Isc_tab') || needLUT('Voc_tab')             % Se mancano le LUT necessarie
        error(['Mancano G_bp/Isc_tab/Voc_tab nel base workspace. ' ...
               'Creare le LUT prima di chiamare runSimGetVI (vedi blocco PREPARA LUT).']); % Errore esplicativo
    end

    % --- Parametri per il modello
    assignin('base','G',   [0 Gval; 1 Gval]);  % Passa G come segnale costante [tempo valore]
    assignin('base','Voc', Voc);               % Passa Voc al modello
    assignin('base','Isc', Isc);               % Passa Isc al modello
    assignin('base','n',   n_id);              % Passa n (idealità)
    assignin('base','Rs',  Rs);                % Passa Rs
    assignin('base','Rsh', Rsh);               % Passa Rsh

    % --- Sopprimi warning/issue dei From Workspace (no-op se il modello non ha questi blocchi)
    setFromWorkspaceST0(modelName);            % Imposta Sample Time = 0 per i From Workspace (se presenti)

    % --- Esecuzione simulazione (ritorna oggetto 'out' con segnali nominati)
    out = sim(modelName, 'ReturnWorkspaceOutputs','on'); % Esegue simulazione e ritorna outputs in 'out'

    % --- Estrai i segnali V,I (usa helper che gestisce diversi tipi: timeseries, numeric, timetable)
    Vsim = fetch_signal_vector('VoutPV', out); % Estrae vettore V simulato
    Isim = fetch_signal_vector('IoutPV', out); % Estrae vettore I simulato

    % --- Pulizia NaN/Inf
    Vsim = Vsim(:); Isim = Isim(:);            % Forza vettori colonna
    ok = isfinite(Vsim) & isfinite(Isim);      % Maschera di valori finiti
    Vsim = Vsim(ok); Isim = Isim(ok);          % Filtra non-finiti
end

function r = residual_IV_for_fit_percurve(x, modelName, Vexp, Iexp, Gval, Voc, Isc_eff, lambdaFF, VMIN_FIT)
% residual_IV_for_fit_percurve — Restituisce il vettore residuo per lsqnonlin.             % Descrizione
% Componenti del residuo:
%   1) Scarto sui punti I(V) sperimentali (per V >= VMIN_FIT), pesato più dove I è piccola; % Spiega la prima parte del residuo
%   2) (opzionale) Penalità su differenza di Fill Factor (FF_sim - FF_exp) scalata da lambdaFF. % Spiega la componente FF
    
    % Simula con i parametri correnti x = [n, Rs, Rsh]
    [Vsim, Isim] = runSimGetVI(modelName, Gval, Voc, Isc_eff, x(1), x(2), x(3)); % Simula il modello con parametri correnti
    [Vsim, Isim] = uniqueXYavg(Vsim, Isim);                                      % Ordina e media duplicati

    if numel(Vsim) < 3                        % Se troppo pochi punti simulati
        r = 1e3 * ones(size(Iexp));           % Restituisci residui grandi (spinge l’ottimizzatore altrove)
        return;                                % Esci dalla funzione
    end

    % --- Seleziona sottoinsieme “stabile” per il fit (evita V ~ 0)
    mexp = isfinite(Vexp) & isfinite(Iexp) & (Vexp >= VMIN_FIT); % Maschera per punti sperimentali sopra soglia V
    if ~any(mexp)                           % Se nessun punto valido
        r = 1e3 * ones(size(Iexp));         % Penalizza fortemente
        return;                             % Esci
    end
    Vq = Vexp(mexp);                        % Vettore tensioni dove interpolare la simulazione

    % --- Interpola I_sim nei punti V sperimentali (pchip robusto, fallback linear)
    try
        Isim_on_exp = interp1(Vsim, Isim, Vq, 'pchip', 'extrap'); % Interpola con pchip per forma morbida
    catch
        Isim_on_exp = interp1(Vsim, Isim, Vq, 'linear', 'extrap'); % Fallback lineare se pchip fallisce
    end

    % --- Pesi: enfatizza regioni a bassa corrente (tipicamente più informative)
    w = 1 ./ max(1e-9, abs(Iexp(mexp)));      % Pesi inversamente proporzionali alla corrente (evita divisione per zero)

    % --- Al buio: dai più peso alla regione vicino a Voc (comportamento diodo)
    if Gval <= 0 && isfinite(Voc) && Voc > 0  % Se dark e Voc valido
        alphaV = 3;                            % Esponente che accentua pesi vicino a Voc
        w = w .* max( (Vq./Voc).^alphaV, 1e-2 ); % Scala i pesi con funzione di V/Voc (min 1e-2 per non annullare)
    end

    % --- Residuo base (scarto sulla corrente normalizzato dai pesi)
    rI = (Isim_on_exp - Iexp(mexp)) .* sqrt(w); % Residuo pesato punto-a-punto

    % --- Penalità sul Fill Factor (solo in luce e se definibili)
    if lambdaFF > 0 && isfinite(Voc) && Voc > 0 && abs(Isc_eff) > 0        % Se attiva e definibile
        FF_sim = ff_from_curve(Vsim, Isim, Voc, Isc_eff);                  % Calcola FF dalla curva simulata
        FF_exp = ff_from_curve(Vexp(mexp), Iexp(mexp), Voc, Isc_eff);      % Stima FF dalla curva sperimentale (stessi punti selezionati)
        if isfinite(FF_sim) && isfinite(FF_exp)                            % Se entrambi finiti
            r = [rI; lambdaFF * (FF_sim - FF_exp)];                        % Aggiunge componente FF al vettore residuo
            return                                                         % Esce
        end
    end

    r = rI;  % solo componente sui punti se penalità FF non applicabile
end

function FF = ff_from_curve(V, I, Voc, Isc)
% ff_from_curve — Calcola il Fill Factor da una curva I–V discreta.          % Descrizione
% Definizione: FF = Pmax / (Voc * |Isc|) con Pmax = max(V.*|I|).             % Formula
    if ~(isfinite(Voc) && isfinite(Isc)) || Voc <= 0 || abs(Isc) <= 1e-9     % Controlli di validità
        FF = NaN; 
        return;
    end
    V = V(:); I = I(:);                         % Vettori colonna
    ok = isfinite(V) & isfinite(I);             % Maschera finiti
    V = V(ok); I = I(ok);                       % Filtra non-finiti
    if numel(V) < 2, FF = NaN; return; end      % Necessari almeno 2 punti

    [V, I] = uniqueXYavg(V, I);                 % Ordina e media duplicati
    P = V .* abs(I);                            % Potenza istantanea
    Pmax = max(P, [], 'omitnan');               % Massimo della potenza
    FF = Pmax / (Voc * abs(Isc));               % Normalizzazione per definizione di FF
end

function [xU, yU] = uniqueXYavg(x, y)
% uniqueXYavg — Ordina per x crescente, raggruppa duplicati e ne fa la media su y. % Descrizione
    x = x(:); y = y(:);                        % Vettori colonna
    ok = isfinite(x) & isfinite(y);            % Maschera finiti
    x = x(ok); y = y(ok);                      % Filtra
    [x, ord] = sort(x);                        % Ordina x crescente e ottieni l’ordine
    y = y(ord);                                % Riordina y coerentemente
    [xU, ~, ic] = unique(x, 'stable');         % Trova valori unici preservando l’ordine
    yU = accumarray(ic, y, [], @mean);         % Media dei corrispondenti y per duplicati
end

function vec = fetch_signal_vector(varName, out)
% fetch_signal_vector — Estrae un vettore numerico dal base workspace o da 'out'. % Descrizione
% Accetta:
%   - timeseries (usa .Data)
%   - numeric matrix [t y] (prende la 2a colonna), o vettore (prende così com’è)
%   - timetable (prende la prima colonna)
% Ordine di ricerca: base workspace -> 'out' (WorkspaceOutputs della simulazione).
    vec = [];                                   % Inizializza output
    try
        % 1) Prova dal base workspace (alcuni modelli lasciano variabili globali)
        if evalin('base', sprintf('exist(''%s'',''var'')', varName)) == 1      % Se la variabile esiste nel base
            val = evalin('base', varName);                                     % Recupera la variabile
            vec = local_cast_vector(val);                                      % Converte in vettore numerico se possibile
            if ~isempty(vec), return; end                                      % Se ottenuto vettore, esci
        end
        % 2) Prova dall'oggetto 'out' ritornato da sim()
        if nargin >= 2 && ~isempty(out)                                        % Se 'out' è disponibile
            try
                ts = out.get(varName);                                         % Tenta di leggere il segnale per nome
                vec = local_cast_vector(ts);                                   % Converte in vettore numerico
            catch
                % ignorabile: variabile non presente in out                   % Se non presente, prosegue
            end
        end
    catch
        vec = [];                                                               % In caso di errore, ritorna vuoto
    end

    function v = local_cast_vector(val)                                         % Helper nested per normalizzare il tipo
        v = [];
        if isa(val,'timeseries')                                                % Se timeseries
            v = val.Data(:);                                                    % Prendi i dati come vettore colonna
        elseif isnumeric(val)                                                   % Se numerico
            if ismatrix(val) && size(val,2)>=2                                  % Se matrice [t y] o simile
                v = val(:,2);                                                   % Prendi la seconda colonna come segnale
            else
                v = val(:);                                                     % Altrimenti vettorializza
            end
        elseif istimetable(val)                                                 % Se timetable
            v = val{:,1}; v = v(:);                                             % Prendi la prima colonna come vettore
        end
    end
end

function setFromWorkspaceST0(modelName)
% setFromWorkspaceST0 — Tenta di impostare i From Workspace su 'Sample time = 0' % Descrizione
% e disabilitare warning fastidiosi. Se Simulink non è disponibile o il modello
% non contiene tali blocchi, non fa nulla (fail-safe).
    try
        if ~bdIsLoaded(modelName), load_system(modelName); end  % Carica il modello se non già caricato
        blks = find_system(modelName, 'BlockType','FromWorkspace'); % Trova tutti i blocchi From Workspace
        for i = 1:numel(blks)                                     % Cicla sui blocchi trovati
            try
                set_param(blks{i}, 'SampleTime', '0');            % Imposta Sample Time a 0 (ereditato dal solver)
                set_param(blks{i}, 'Interpolate', 'on');          % Abilita interpolazione lineare
                % Formato dati struttura con campi time/values opzionale:
                % qui lasciamo il default, usiamo [t u].            % Nota operativa
            catch
                % ignora blocchi protetti/locked                    % Se blocco non modificabile, passa oltre
            end
        end
    catch
        % Simulink non disponibile o modello non accessibile: nessuna azione necessaria % Fail-safe: non solleva errore
    end
end

function s = exitflag_to_text(exitflag)
% exitflag_to_text — Converte l’exitflag numerico in un breve testo esplicativo. % Descrizione
    switch exitflag
        case {1,2,3,4}
            s = 'Local minimum found.';                       % Minimo locale trovato
        case 0
            s = 'Stopped: max iterations/func-count.';        % Stop per limiti su iterazioni/valutazioni
        case -1
            s = 'Stopped by output/plot function.';           % Stop da funzione esterna
        case -2
            s = 'No feasible point found.';                   % Nessun punto ammissibile
        otherwise
            s = sprintf('Stopped (exitflag=%d).', exitflag);  % Caso generico con codice
    end
end
